<?php
/*
Plugin Name: XAI Schema Markup
Description: Automatically adds Article, Organization, Person, BreadcrumbList, and FAQPage schemas to WordPress posts and pages, with FAQ detection and ISO 8601 compliant dates.
Version: 1.2
Author: Grok
License: GPL2
*/

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Helper function to convert Jalali date to Gregorian and format as ISO 8601
function xai_schema_convert_to_iso8601($post_id, $type = 'publish') {
    // Get WordPress timezone
    $timezone = new DateTimeZone(wp_timezone_string());
    
    // Check if WP-Parsidate is active
    if (function_exists('parsidate')) {
        // Get post date (publish or modified)
        $date = ($type === 'publish') ? get_post_time('Y/m/d H:i:s', false, $post_id) : get_post_modified_time('Y/m/d H:i:s', false, $post_id);
        
        // Convert Jalali to Gregorian using WP-Parsidate
        $gregorian = parsidate('Y-m-d H:i:s', $date, 'eng');
        $date_obj = new DateTime($gregorian, $timezone);
    } else {
        // Use WordPress default (Gregorian) date
        $date = ($type === 'publish') ? get_post_time('Y-m-d H:i:s', false, $post_id) : get_post_modified_time('Y-m-d H:i:s', false, $post_id);
        $date_obj = new DateTime($date, $timezone);
    }
    
    // Format as ISO 8601
    return $date_obj->format('c');
}

// Add admin settings page
add_action('admin_menu', 'xai_schema_add_settings_page');
function xai_schema_add_settings_page() {
    add_options_page(
        'XAI Schema Settings',
        'XAI Schema',
        'manage_options',
        'xai-schema-settings',
        'xai_schema_settings_page'
    );
}

// Settings page HTML
function xai_schema_settings_page() {
    ?>
    <div class="wrap">
        <h1>XAI Schema Markup Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('xai_schema_settings');
            do_settings_sections('xai_schema_settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Register settings
add_action('admin_init', 'xai_schema_register_settings');
function xai_schema_register_settings() {
    register_setting('xai_schema_settings', 'xai_schema_logo_url');
    register_setting('xai_schema_settings', 'xai_schema_social_links');

    add_settings_section(
        'xai_schema_main',
        'Schema Settings',
        null,
        'xai_schema_settings'
    );

    add_settings_field(
        'xai_schema_logo_url',
        'Logo URL',
        'xai_schema_logo_url_callback',
        'xai_schema_settings',
        'xai_schema_main'
    );

    add_settings_field(
        'xai_schema_social_links',
        'Social Media Links (one per line)',
        'xai_schema_social_links_callback',
        'xai_schema_settings',
        'xai_schema_main'
    );
}

function xai_schema_logo_url_callback() {
    $value = get_option('xai_schema_logo_url', '');
    echo '<input type="url" name="xai_schema_logo_url" value="' . esc_attr($value) . '" class="regular-text" />';
}

function xai_schema_social_links_callback() {
    $value = get_option('xai_schema_social_links', '');
    echo '<textarea name="xai_schema_social_links" rows="5" class="large-text">' . esc_textarea($value) . '</textarea>';
}

// Add schema to wp_head
add_action('wp_head', 'xai_schema_add_markup');
function xai_schema_add_markup() {
    global $post;

    // Organization Schema (only on homepage)
    if (is_front_page()) {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => get_bloginfo('name'),
            'url' => home_url(),
            'logo' => get_option('xai_schema_logo_url', ''),
            'sameAs' => array_filter(explode("\n", get_option('xai_schema_social_links', '')))
        );
        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
    }

    // Article, Person, BreadcrumbList, and FAQPage for single posts
    if (is_single() && isset($post)) {
        // Article Schema
        $author_id = $post->post_author;
        $author = get_the_author_meta('display_name', $author_id);
        $image = wp_get_attachment_image_src(get_post_thumbnail_id($post->ID), 'full');
        $image = $image ? $image[0] : '';

        $article_schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => get_the_title($post->ID),
            'author' => array(
                '@type' => 'Person',
                'name' => $author,
                'url' => get_author_posts_url($author_id)
            ),
            'datePublished' => xai_schema_convert_to_iso8601($post->ID, 'publish'),
            'dateModified' => xai_schema_convert_to_iso8601($post->ID, 'modified'),
            'image' => $image,
            'publisher' => array(
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                'logo' => array(
                    '@type' => 'ImageObject',
                    'url' => get_option('xai_schema_logo_url', '')
                )
            ),
            'mainEntityOfPage' => array(
                '@type' => 'WebPage',
                '@id' => get_permalink($post->ID)
            )
        );

        // BreadcrumbList Schema
        $categories = get_the_category($post->ID);
        $breadcrumb_items = array(
            array(
                '@type' => 'ListItem',
                'position' => 1,
                'name' => 'Home',
                'item' => home_url()
            )
        );

        if (!empty($categories)) {
            $breadcrumb_items[] = array(
                '@type' => 'ListItem',
                'position' => 2,
                'name' => $categories[0]->name,
                'item' => get_category_link($categories[0]->term_id)
            );
        }

        $breadcrumb_items[] = array(
            '@type' => 'ListItem',
            'position' => count($breadcrumb_items) + 1,
            'name' => get_the_title($post->ID),
            'item' => get_permalink($post->ID)
        );

        $breadcrumb_schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => $breadcrumb_items
        );

        // FAQPage Schema (if "سوالات متداول" or "FAQ" in title or content)
        $faq_schema = array();
        $content = $post->post_content;
        $title = get_the_title($post->ID);
        if (stripos($title, 'سوالات متداول') !== false || stripos($title, 'FAQ') !== false ||
            stripos($content, 'سوالات متداول') !== false || stripos($content, 'FAQ') !== false) {
            $faq_items = array();
            $dom = new DOMDocument();
            @$dom->loadHTML('<?xml encoding="UTF-8">' . $content); // Suppress warnings for malformed HTML
            $xpath = new DOMXPath($dom);
            $headings = $xpath->query('//h2|//h3'); // Assuming questions are in h2 or h3

            foreach ($headings as $index => $heading) {
                $question = trim($heading->textContent);
                $answer = '';
                $next_node = $heading->nextSibling;
                while ($next_node && $next_node->nodeName !== 'h2' && $next_node->nodeName !== 'h3') {
                    if ($next_node->nodeType === XML_TEXT_NODE || $next_node->nodeName === 'p') {
                        $answer .= $next_node->textContent;
                    }
                    $next_node = $next_node->nextSibling;
                }
                $answer = trim($answer);
                if ($question && $answer) {
                    $faq_items[] = array(
                        '@type' => 'Question',
                        'name' => $question,
                        'acceptedAnswer' => array(
                            '@type' => 'Answer',
                            'text' => $answer
                        )
                    );
                }
            }

            if (!empty($faq_items)) {
                $faq_schema = array(
                    '@context' => 'https://schema.org',
                    '@type' => 'FAQPage',
                    'mainEntity' => $faq_items
                );
            }
        }

        // Output schemas
        echo '<script type="application/ld+json">' . wp_json_encode($article_schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
        echo '<script type="application/ld+json">' . wp_json_encode($breadcrumb_schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
        if (!empty($faq_schema)) {
            echo '<script type="application/ld+json">' . wp_json_encode($faq_schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
        }
    }
}
?>